<?php

namespace App\Observers;

use File;
use Carbon\Carbon;
use App\Helper\Files;
use App\Models\Invoice;
use App\Models\Estimate;
use Illuminate\Support\Str;
use App\Models\EstimateItem;
use App\Models\InvoiceItems;
use App\Models\Notification;
use App\Models\UniversalSearch;
use App\Events\NewEstimateEvent;
use App\Models\EstimateItemImage;
use App\Traits\UnitTypeSaveTrait;
use App\Events\EstimateAcceptedEvent;
use App\Events\EstimateDeclinedEvent;
use App\Models\EstimateTemplateItemImage;

class EstimateObserver
{

    use UnitTypeSaveTrait;

    public function saving(Estimate $estimate)
    {
        $this->unitType($estimate);

        if (!isRunningInConsoleOrSeeding()) {

            if (\user()) {
                $estimate->last_updated_by = user()->id;
            }

            if (request()->has('calculate_tax')) {
                $estimate->calculate_tax = request()->calculate_tax;
            }
        }

    }

    public function creating(Estimate $estimate)
    {
        $estimate->hash = md5(microtime());

        if (\user()) {
            $estimate->added_by = user()->id;
        }

        if (request()->type && (request()->type == 'save' || request()->type == 'draft')) {
            $estimate->send_status = 0;
        }

        if (request()->type == 'draft') {
            $estimate->status = 'draft';
        }

        if (company()) {
            $estimate->company_id = company()->id;
        }
    }

    public function created(Estimate $estimate)
    {

        if (!isRunningInConsoleOrSeeding()) {
            if (!empty(request()->item_name)) {

                $itemsSummary = request()->item_summary;
                $cost_per_item = request()->cost_per_item;
                $hsn_sac_code = request()->hsn_sac_code;
                $quantity = request()->quantity;
                $amount = request()->amount;
                $tax = request()->taxes;
                $invoice_item_image = request()->invoice_item_image;
                $invoice_item_image_delete = request()->invoice_item_image_delete;
                $invoice_item_image_url = request()->invoice_item_image_url;
                $invoiceOldImage = request()->image_id;
                $invoiceTemplateImage = request()->templateImage_id;

                foreach (request()->item_name as $key => $item) :
                    if (!is_null($item)) {
                        $estimateItem = EstimateItem::create(
                            [
                                'estimate_id' => $estimate->id,
                                'item_name' => $item,
                                'item_summary' => $itemsSummary[$key],
                                'type' => 'item',
                                'hsn_sac_code' => (isset($hsn_sac_code[$key]) && !is_null($hsn_sac_code[$key])) ? $hsn_sac_code[$key] : null,
                                'quantity' => $quantity[$key],
                                'unit_price' => round($cost_per_item[$key], 2),
                                'amount' => round($amount[$key], 2),
                                'taxes' => $tax ? array_key_exists($key, $tax) ? json_encode($tax[$key]) : null : null
                            ]
                        );


                        /* Invoice file save here */

                        if ((isset($invoice_item_image[$key]) && $invoice_item_image[$key] != 'yes') || isset($invoice_item_image_url[$key]))
                        {
                            EstimateItemImage::create(
                                [
                                    'estimate_item_id' => $estimateItem->id,
                                    'filename' => !isset($invoice_item_image_url[$key]) ? $invoice_item_image[$key]->getClientOriginalName() : '',
                                    'hashname' => !isset($invoice_item_image_url[$key]) ? Files::uploadLocalOrS3($invoice_item_image[$key], EstimateItemImage::FILE_PATH . '/' . $estimateItem->id . '/') : '',
                                    'size' => !isset($invoice_item_image_url[$key]) ? $invoice_item_image[$key]->getSize() : '',
                                    'external_link' => $invoice_item_image_url[$key] ?? ''
                                ]
                            );

                        }

                        $image = true;

                        if(isset($invoice_item_image_delete[$key]))
                        {
                            $image = false;
                        }

                        if($image && (isset(request()->image_id[$key]) && $invoiceOldImage[$key] != ''))
                        {
                            $estimateOldImg = EstimateItemImage::with('item')->where('id', request()->image_id[$key])->first();

                            $this->duplicateImageStore($estimateOldImg, $estimateItem);
                        }

                        if($image && (isset(request()->templateImage_id[$key]) && $invoiceTemplateImage[$key] != ''))
                        {
                            $estimateTemplateImg = EstimateTemplateItemImage::where('id', request()->templateImage_id[$key])->first();

                            $this->duplicateTemplateImageStore($estimateTemplateImg, $estimateItem);
                        }

                    }

                endforeach;
            }



            if (request()->type != 'save' && request()->type != 'draft') {
                event(new NewEstimateEvent($estimate));
            }
        }
    }

    public function updated(Estimate $estimate)
    {
        if (!isRunningInConsoleOrSeeding()) {
            if ($estimate->status == 'declined') {
                event(new EstimateDeclinedEvent($estimate));
            }
            elseif ($estimate->status == 'accepted') {
                event(new EstimateAcceptedEvent($estimate));
            }
        }
    }

    public function deleting(Estimate $estimate)
    {
        $universalSearches = UniversalSearch::where('searchable_id', $estimate->id)->where('module_type', 'estimate')->get();

        if ($universalSearches) {
            foreach ($universalSearches as $universalSearch) {
                UniversalSearch::destroy($universalSearch->id);
            }
        }

        $notifyData = ['App\Notifications\NewEstimate'];
        \App\Models\Notification::deleteNotification($notifyData, $estimate->id);

    }

    /**
     * duplicateImageStore
     *
     * @param  mixed $estimateOldImg
     * @param  mixed $estimateItem
     * @return void
     */
    public function duplicateImageStore($estimateOldImg, $estimateItem)
    {
        if(!is_null($estimateOldImg)) {

            $file = new EstimateItemImage();

            $file->estimate_item_id = $estimateItem->id;

            $fileName = Files::generateNewFileName($estimateOldImg->filename);

            Files::copy(EstimateItemImage::FILE_PATH . '/' . $estimateOldImg->item->id . '/' . $estimateOldImg->hashname, EstimateItemImage::FILE_PATH . '/' . $estimateItem->id . '/' . $fileName);

            $file->filename = $estimateOldImg->filename;
            $file->hashname = $fileName;
            $file->size = $estimateOldImg->size;
            $file->save();

        }
    }

    public function duplicateTemplateImageStore($estimateTemplateImg, $estimateItem)
    {
        if(!is_null($estimateTemplateImg)) {

            $file = new EstimateItemImage();

            $file->estimate_item_id = $estimateItem->id;

            $fileName = Files::generateNewFileName($estimateTemplateImg->filename);

            Files::copy(EstimateItemImage::FILE_PATH . '/' . $estimateTemplateImg->id . '/' . $estimateTemplateImg->hashname, EstimateTemplateItemImage::FILE_PATH . '/' . $estimateItem->id . '/' . $fileName);

            $file->filename = $estimateTemplateImg->filename;
            $file->hashname = $fileName;
            $file->size = $estimateTemplateImg->size;
            $file->save();

        }
    }

}
